<?php

namespace Wizzard\MagicForger;

class ConfigHelper
{
    // Config array
    public static $config = [];

    // Config file name constant
    public const CONFIG_FILE_NAME = 'mf_config.php';

    // Config path variable
    public static $config_path;

    /**
     * Set up configuration path
     *
     * @param string $base_path
     */
    public static function setup_config_path(string $base_path)
    {
        self::$config_path = $base_path.'/'.self::CONFIG_FILE_NAME;
    }

    /**
     * Get configuration path
     *
     * @return string
     */
    protected static function get_config_path()
    {
        return self::$config_path;
    }

    /**
     * Write configuration into a file
     */
    public static function write_config()
    {
        $path = self::get_config_path();
        $str = '<?php
                        return '.self::varexport(self::$config, true).';';

        file_put_contents($path, $str);
        // After writing the file, format it
        self::format_file($path);
    }

    /**
     * Read configuration from a file
     */
    public static function read_config()
    {
        $path = self::get_config_path();
        self::$config = include $path;
    }

    /**
     * Print configuration
     */
    public static function print_config()
    {
        self::varexport(self::$config);
    }

    /**
     * PHP var_export() function with short array syntax (square brackets) indented 2 spaces.
     *
     * NOTE: The only issue is when a string value has `=>\n[`, it will get converted to `=> [`
     *
     * @see https://www.php.net/manual/en/function.var-export.php
     *
     * @param mixed $expression
     * @param boolean $return
     * @return mixed
     */
    public static function varexport($expression, $return = false)
    {
        $export = var_export($expression, true);
        $patterns = [
            "/array \(/" => '[',
            "/^([ ]*)\)(,?)$/m" => '$1]$2',
            "/=>[ ]?\n[ ]+\[/" => '=> [',
            "/([ ]*)(\'[^\']+\') => ([\[\'])/" => '$1$2 => $3',
        ];
        $export = preg_replace(array_keys($patterns), array_values($patterns), $export);
        if ((bool) $return) {
            return $export;
        } else {
            echo $export;
        }
    }

    /**
     * Format the given file
     *
     * @param string $path
     */
    protected static function format_file(string $path)
    {
        exec('php-cs-fixer fix '.$path);
    }

    /**
     * Set up tables
     */
    public static function set_up_tables()
    {
        $schema = \DB::connection()->getDoctrineSchemaManager();
        // get all the tables available in the database
        $tables = collect($schema->listTableNames())->all();

        $insert_tables = [];
        foreach ($tables as $table) {
            $columns = [];
            $table_columns = $schema->introspectTable($table)->getColumns();

            foreach ($table_columns as $column) {
                $full_class = get_class($column->getType());
                $class_parts = explode('\\', $full_class);
                $class_name = end($class_parts);

                $columns[$column->getName()] = [
                                'type' => $class_name,
                                'should_insert' => [
                                        'controller' => true,
                                        'model' => true,
                                        'requests' => true,
                                        'views' => true,
                                      ],
                                ];
            }

            $insert_tables[$table] = [];
            $insert_tables[$table]['columns'] = $columns;
            $insert_tables[$table]['type'] = 'default';
        }
        // Merge the new tables configuration into the initial config
        self::merge_array_priority(self::$config['tables'], $insert_tables);

        return $tables;
    }

    /**
     * Merge two arrays and ensure priority values do not get overwritten
     *
     * @param array $priority
     * @param array $merged
     */
    private static function merge_array_priority(&$priority, $merged)
    {
        foreach ($merged as $key => $value) {
            // if the priority key is not set, automatically add the merged values
            if (!isset($priority[$key])) {
                $priority[$key] = $value;
            } else {
                // if the value is an array recursively merge with priority
                if (is_array($value) && is_array($priority[$key])) {
                    self::merge_array_priority($priority[$key], $value);
                }
            }
        }
    }
}
