<?php

namespace Wizard\MagicForger\Replacer;

trait TableReplacer
{
    protected ?array $columns = null;

    protected array $columns_to_ignore = [
        'id',
        'created_at',
        'updated_at',
        'created_by',
        'updated_by',
        'deleted_at',
    ];

    /**
     * Retrieve columns for the current table.
     */
    protected function get_columns(): array
    {
        if (is_null($this->columns)) {
            $this->columns = $this->getTableColumns($this->getCurrentTable());
        }

        return $this->columns;
    }

    /**
     * Get a string representation of values for creation.
     */
    protected function getValuesForCreation(): string
    {
        $insert = '';
        foreach ($this->get_columns() as $column) {
            $column_name = $column['name'];
            $insert .= sprintf('$item->%s = $validated["%s"] ?? NULL;', $column_name, $column_name)."\n";
        }

        return $insert;
    }

    /**
     * Get a string representation of table attributes.
     */
    protected function getCasts(): string
    {
        $insert = '';
        foreach ($this->get_columns() as $column) {
            if (in_array($column['name'], $this->columns_to_ignore)) {
                continue;
            }
						$type = $column['type_name'];
						//date
						if(in_array($type, ['date'])) {
							$insert .= sprintf("'%s' => 'date:Y-m-d',", $column['name'])."\n";
						}
						//time
						if(in_array($type, ['timestamp'])) {
							$insert .= sprintf("'%s' => 'date:Y-m-d H:i',", $column['name'])."\n";
						}
        }

        return $insert;
    }

    /**
     * Get a string representation of table attributes.
     */
    protected function getAttributes(): string
    {
        $insert = '';
        foreach ($this->get_columns() as $column) {
            if (in_array($column['name'], $this->columns_to_ignore)) {
                continue;
            }
            $insert .= sprintf("'%s' => '',", $column['name'])."\n";
        }

        return $insert;
    }

    /**
     * Get a string representation of table fillable columns.
     */
    protected function getFillable(): string
    {
        $insert = '';
        foreach ($this->get_columns() as $column) {
            if (in_array($column['name'], $this->columns_to_ignore)) {
                continue;
            }
            $insert .= sprintf("'%s',", $column['name'])."\n";
        }

        return $insert;
    }

    /**
     * Get formatted validation rules for table columns.
     */
    protected function getValuesForValidation(): string
    {
        $insert = '';
        foreach ($this->get_columns() as $column) {
            if (in_array($column['name'], $this->columns_to_ignore)) {
                continue;
            }
            $insert .= sprintf("'%s' => 'nullable',", $column['name'])."\n";
        }

        return $insert;
    }

    /**
     * Apply insertions in the target template.
     */
    public function apply_inserts(string $target): string
    {
        $inserts = $this->get_all_keywords($target);
        $available_insertions = $this->get_available_inserts();

        return str_replace(
            array_keys($available_insertions),
            $available_insertions,
            $target
        );
    }

    /**
     * Get available insertion points for the template.
     */
    public function get_available_inserts(): array
    {
        return [
            '// {{ valuesForCreation }}' => $this->getValuesForCreation(),
            '# {{ attributeInsertPoint }}' => $this->getAttributes(),
            '# {{ castInsertPoint }}' => $this->getCasts(),
            '# {{ fillableInsertPoint }}' => $this->getFillable(),
            '// {{ valuesForValidation }}' => $this->getValuesForValidation(),
        ];
    }
}
